
/***********************************************************************************************
 * @file    bch.c
 * @version $Rev:: 4837         $
 * @date    $Date:: 2022-09-02 #$
 * @brief   This file provides all the RF firmware functions.
 ***********************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ***********************************************************************************************/

/* Includes -----------------------------------------------------------------------------------*/
#include	<stdio.h>
#include <string.h>
#include	"ht32.h"
#include	"bc3603.h"
#include	"bch.h"


typedef union
{
	u32	v;
	struct
	{
		u32	b0: 8;
		u32	b1: 8;
		u32	b2: 8;
		u32	b3: 8;
	}b;
} U32B;

typedef union
{
	u16	v;
	struct
	{
		u16	b0: 8;
		u16	b1: 8;
	}b;
} U16B;

#define	BCH16_MASK			(u16)0xFFE0
#define	G16					(u16)0xB400
#define	P16					(u16)0xF134

#define	BCH32_MASK			(u32)0xFFFFFFC0
#define	G32					(u32)0xDE000000
#define	P32					(u32)0x864FB8AC

void BCH32_264T1(u32 lap, u8 *pIdcode)
{
	u8		iteration_mod;
	u32	a,g,mask;
	U32B	tmp;
	
	a = lap;	
	a &= ((((u32)1) << LAP_32_K) - 1);
	if(a & (((u32)1) << (LAP_32_K - 1)))	
	{ a |= (BARKER_PAD_1 << LAP_32_K); }
	else 													
	{ a |= (BARKER_PAD_0 << LAP_32_K); }

	a <<= (BCH_32_N - BCH_32_K);
	
	tmp.v = (a ^ P32) & BCH32_MASK;
	g = G32;
	mask = 0x80000000;	
	
	for (iteration_mod = 0; iteration_mod<BCH_32_K; iteration_mod++)
	{
		if(tmp.v & mask)	tmp.v ^= g;
		mask >>= 1;
		g >>= 1;
	}
	tmp.v = (a & BCH32_MASK) | ((tmp.v ^ P32) & ~BCH32_MASK);
	
	pIdcode[0] = tmp.b.b3;
	pIdcode[1] = tmp.b.b2;
	pIdcode[2] = tmp.b.b1;
	pIdcode[3] = tmp.b.b0;
}

void BCH16_114T1(u32 lap, u8 *pIdcode)
{
	u8		iteration_mod;
	u16	a,g,mask;
	U16B	tmp;
		
	lap &= ((((u16)1) << LAP_16_K) - 1);
	if (lap & (((u16)1) << (LAP_16_K - 1))) 
	{ lap |= (BARKER_PAD_1 << LAP_16_K); }
	else 
	{ lap |= (BARKER_PAD_0 << LAP_16_K); }
	a = lap & 0xFFFF;
	a <<= (BCH_16_N - BCH_16_K);
	g = G16;
	mask = 0x8000;
	tmp.v = (a ^ P16) & BCH16_MASK;
	
	for(iteration_mod = 0; iteration_mod < BCH_16_K; iteration_mod++)
	{
		if(tmp.v & mask)	{ tmp.v ^= g;	}
		g >>= 1;
		mask >>= 1;
	}
	tmp.v = (a & BCH16_MASK) | ((tmp.v ^ P16) & ~BCH16_MASK);
	pIdcode[0] = tmp.b.b1;
	pIdcode[1] = tmp.b.b0;
}
/***********************************************************************************************
 * @brief syncword calculate of Bose-Chaudhuri-Hocquenghem(BCH) codes
 * @param SNL: Selection of syncword length
 *   This parameter can be one of below:
 *     @arg SYNCLEN4BYTE: syncword length 4bytes
 *     		SYNCLEN6BYTE: syncword length 6bytes
 *     		SYNCLEN8BYTE: syncword length 8bytes
 * @param lap: Pointer to the device address or serial number buffer,
 *				   must be greater than or equal to 40bit(5bytes)
 *     @arg SYNCLEN4BYTE: Take 20bit to calculate syncword
 *     		SYNCLEN6BYTE: Take 25bit to calculate syncword
 *     		SYNCLEN8BYTE: Take 40bit to calculate syncword
 * @param pIdcode: Pointer to a syncword buffer,must be greater than or equal to 64bit(8bytes)
 * @retval None
 ***********************************************************************************************/
void BCH_SyncwordCalculate(RF_SYNCLEN_Enum SNL, u8 *lap, u8 *pIdcode)
{
	u8		pid[4],x;
	u32	lap32;
	
	for(x=0;x<SYNCWORD_MAX_SIZE;x++) pIdcode[x] = 0;
	lap32 = lap[0] + (lap[1] << 8) + (lap[2] << 16);
	BCH32_264T1(lap32, pid);
	for(x=0;x<4;x++) pIdcode[x] = pid[x];	
	switch(SNL)
	{
		case SYNCLEN5BYTE:		
		case SYNCLEN6BYTE:
			lap32 = lap[2] + (lap[3] << 8) + (lap[3] << 16) ;
			lap32 >>= 4;
			BCH16_114T1(lap32, pid);
			pIdcode[4] = pid[0];
			pIdcode[5] = pid[1];				
			break;
		case SYNCLEN7BYTE:		
		case SYNCLEN8BYTE:
			lap32 = lap[2] + lap[3]*256 + lap[4]*65536;
			lap32 >>= 4;
			BCH32_264T1(lap32, pid);
			for(x=0;x<4;x++) pIdcode[x+4] = pid[x];
			break;
		default:
			break;		
	}
}

